# !/usr/bin/env python3
# -*- coding: utf-8 -*-

#*******************************************************************************
# File : synchronized_pump.py
# Package : AMFTools example
# Description : This module provide an example of how to pump with 2 AMF SPM product at the same time (synchronized)
# Author : Paul Giroux AMF (paul.giroux@amf.ch)
# Date Created : november 07, 2023
# Date Modified : april 12, 2024
# Version : 0.1.5
# Python Version : 3.11.4
# Dependencies : pyserial, ftd2xx
# License : all Right reserved : Proprietary license (Advanced Microfluidics S.A.)
#*******************************************************************************

import amfTools # import the module
import threading

list_amf = amfTools.util.getProductList() # get the list of AMF products connected to the computer
print("\n******* List of AMF products connected to the computer *******")
for amf in list_amf:
    print(amf)
print("**************************************************************\n")

# CREATION OF THE MULTI AMF OBJECT (2 SPM)
class MultiAMF:
    SPM1 : amfTools.AMF = None
    SPM2 : amfTools.AMF = None
    __stop : bool = False

    def __init__(self, SPM1 : amfTools.AMF, SPM2 : amfTools.AMF):
        self.SPM1 = SPM1
        self.SPM2 = SPM2
    
    def wait(self):
        self.SPM1.pullAndWait()
        self.SPM2.pullAndWait()

    def pump(self, position : int):
        self.SPM1.pump(position)
        self.SPM2.pump(position)

    def stop(self):
        self.__stop = True

    def continuousAlternativePump(self):
        while True:
            self.SPM1.pump(0, block=False) #with block=False the function is non blocking (the program continue to run)
            self.SPM2.pump(3000, block=False)
            self.wait() # wait for the 2 pumps to be done
            if self.__stop: return
            self.SPM1.pump(3000, block=False)
            self.SPM2.pump(0, block=False)
            self.wait()
            if self.__stop: return

# Connect to the two first AMF product that is an SPM of the list
SPM1 = None
SPM2 = None
product : amfTools.Device = None
for product in list_amf:
    if product.deviceType == "SPM":
        if SPM1 is None: SPM1 = amfTools.AMF(product)
        elif SPM2 is None: 
            SPM2 = amfTools.AMF(product)
            break

if product is None:
    raise Exception("No SPM product found")

#set the pump parameters
SPM1.setAccelerationRate(1557) # set the aceleration rate to 1557 pulse/s^2
SPM1.setDecelerationRate(59590) # set the deceleration rate to 59590 pulse/s^2
SPM1.setSpeed(150) # set the speed to 150 pulse/s
SPM1.setSyringeSize(1000) # set the syringe size to 1000 µL (1 mL)
SPM2.setDecelerationRate(1557) # set the aceleration rate to 1557 pulse/s^2
SPM2.setDecelerationRate(59590) # set the deceleration rate to 59590 pulse/s^2
SPM2.setSpeed(150) # set the speed to 150 pulse/s
SPM2.setSyringeSize(1000) # set the syringe size to 1000 µL (1 mL)


# check if the products are homed (if not, home it)
if not SPM1.getHomeStatus():
    SPM1.home(block=False) # home the product (non blocking)
if not SPM2.getHomeStatus():
    SPM2.home(block=False) # home the product (non blocking)
#watting for the homing to be done
SPM1.pullAndWait()
SPM2.pullAndWait()

#define the multi amf object
multiAMF = MultiAMF(SPM1, SPM2)
Tread = threading.Thread(target=multiAMF.continuousAlternativePump).start()

input("Press enter to stop the program\n")

multiAMF.stop()
SPM1.hardStop()
SPM2.hardStop()
print("**************************************************************")
print("********************* End of the program *********************")
print("**************************************************************\n")